import Link from "next/link";
import { client } from "@/sanity/lib/client";

const POSTS_QUERY = `*[
  _type == "post"
  && defined(slug.current)
]|order(publishedAt desc)[0...12]{
  _id,
  title,
  slug,
  publishedAt,
  "author": author->name,
  "category": categories[]->{_id, title},
  body
}`;

const options = { next: { revalidate: 30 } };

// Function to get the first 3 lines or roughly 150 characters from the body content
function getExcerpt(body) {
    const text = body
        .map(block => (block.children ? block.children.map(child => child.text).join(" ") : ""))
        .join(" ");
    return text.length > 150 ? `${text.substring(0, 150)}...` : text;
}

export default async function BlogBox() {
    const posts = await client.fetch(POSTS_QUERY, {}, options);
    console.log(posts)

    return (
        <main className="container mx-auto min-h-screen max-w-3xl p-8">
            <h1 className="text-4xl font-bold mb-8">Posts</h1>
            <ul className="flex flex-col gap-y-6">
                {posts.map((post) => (
                    <li className="hover:underline p-4 border rounded-md shadow-lgல" key={post._id}>
                        <Link href={`/blog/${post.slug.current}`}>
                            <div>
                                <h2 className="text-2xl font-semibold">{post.title}</h2>
                                <p className="text-sm text-gray-500">
                                    {new Date(post.publishedAt).toLocaleDateString()} by {post.author} in {post.category.map(cate => <span>{cate.title}</span>)}
                                </p>
                                <p className="text-gray-700 mt-2">
                                    {getExcerpt(post.body)}
                                </p>
                            </div>
                        </Link>
                    </li>
                ))}
            </ul>
        </main>
    );
}
